/************************************************************************
	CAPI.C

	Copyright (c) 1996-2000 Datalight, Inc.
	Confidential and Proprietary
	All Rights Reserved

	Provides the Sockets Compatible API (CAPI)

	History
	1.00 1999-03-18 Starting version control, code usable for Microsoft
		or Borland compilers
	1.01 1999-04-29 fix set_opt() parameter
	1.02 1999-05-27 New version using Datalight Coding guidelines
	1.03 2000-07-11 Added MSVC compatibilaty
	1.10 2000-11-29 pkg Extensive changes to make the code more portable
	1.11 2000-12-15 RWK added function headers
 ************************************************************************/

 /*
        1. To what API is this compatible to?  It does not
        appear to be anywhere near compatible to FTP's TCP/IP
        stack API.  Am I looking in the wrong place?
        2. Still needs acurrate function headers.
        4. There are a couple error cases signified by return
        of a -1 as a pointer.  This is quite dangerous.  Could
        NULL be used instead?
        5. A method to translate a real mode compatible pointer
        into a native pointer is still needed.  This should
        be done by the REALMODE_xxx macros, but might not be
        possible due to a protected mode address being outside
        the lower 1MB space.  Physical addresses could be used
        if the Sockets was capable of using bigreal mode...
        6. The move of data back and forth between real mode
        and the native mode will be difficult.  If the Sockets
        API was consolidated to include only one data pointer
        and kept everything else in registers, it would be
        much simpler.  However, in this implementation, we
        will have a signficant task to maintain multiple
        real mode transfer buffers.
*/
#include "compiler.h"
#include "capi.h"

/*
	Publicly available error variables that futher describe
	the details of failures while talking with the Sockets
	kernel.  These are set immediately following a call into
	the CAPI and are only valid if the previous function
	call indicated a failure.
*/
int iNetErrNo;
int iNetSubErrNo;


/*
	Parameters
	none

	Description
	Returns the version number of the Sockets API

	Returns
	On success, previous state of notification
	On error, -1 if the Sockets API is not found
*/
int GetSocketsVersion(void)
{
	X86Regs r;

	/* Set the function number to call */
	r.ax = GET_SOCKETS_VERSION << 8;
	return CallSocketsDosApi(&r);
}


/*
	Parameters
	none

	Description
	Disables Asynchronous notifications (callbacks).

	Returns
	On success, 0 for disabled, 1 for enabled.
	On error, -1 with both iNetErrNo and iNetSubErr set
*/
int DisableAsyncNotification(void)
{
	X86Regs r;

	/* Set the function number to call */
	r.ax = DISABLE_ASYNC_NOTIFICATION << 8;
	return CallSocketsDosApi(&r);
}


/*
	Parameters
	none

	Description
	Enables asynchronous notifications (callbacks).

	Returns
	On success, previous state of notification
	On error, -1 with both iNetErrNo and iNetSubErr set
*/
int EnableAsyncNotification(void)
{
	X86Regs r;

	r.ax = ENABLE_ASYNC_NOTIFICATION << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
	none

	Description
	Checks a Dos Compatible socket for validity.

	Returns
	On success, 0
	On error, -1 with SocketsErrNo containing the error code.
*/

int IsSocket(int iSocket)
{
	X86Regs r;

	r.bx = iSocket;
	r.ax = IS_SOCKET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
	None

	Description
	Gets a DOS-compatible socket handle.  This function calls DOS to open a 	DOS file handle.

	Returns
	On success, socket handle.
	On error, -1 with SocketsErrNo containing the error code.
*/

int GetDCSocket(void)
{
	X86Regs r;

	r.ax = GET_DC_SOCKET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
	None

	Description
	Gets a socket handle.

	Returns
	On success, socket handle.
	On error, -1 with SocketsErrNo containing the error code.

*/
int GetSocket(void)
{
	X86Regs r;

	r.ax = GET_SOCKET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
psKc
Pointer to KERNEL_CONFIG structure.
bKMaxTcp	Number of TCP sockets allowed.
BKMaxUdp	Number of UDP sockets allowed.
bKMaxIp	Number of IP sockets allowed (0).
bKMaxRaw	Number of RAW_NET sockets allowed (0).
bKActTcp	Number of TCP sockets in use.
bKActUdp	Number of UDP sockets in use.
bKActIp	Number of IP sockets in use (0).
bKActRaw	Number of RAW_NET sockets in use (0).
wKActDCS	Number of active Dos Compatible Sockets.
wKActSoc	Number of active normal Sockets.
bKMaxLnh	Maximum header on an attached network.
bKMaxLnt	Maximum trailer on an attached network.
bKLBUF_SIZE	Size of a large packet buffer.
bKNnet	Number of network interfaces attached.
dwKCticks	Milliseconds since kernel started.
dwKBroadcast	IP broadcast address in use.

Description
Gets the kernel configuration

Returns
On Success, 0 with KERNEL_CONFIG structure filled in
On error, -1 with SocketsErrNo containing the error code.
*/

/*
	Parameters
	None

Description
Gets the version number of the Compatible API.

Returns
On success, 0x214.
On error,  -1 with SocketsErrNo containing the error code.
*/

int GetVersion(void)
{
	X86Regs r;

	r.ax = GET_NET_VERSION << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
none

	Description
Changes a Dos Compatible socket handle into a normal socket handle. This function calls DOS to close a DOS file handle.

Returns
On success, returns socket handle.
On error, -1 with SocketsErrNo containing the error code.
*/
int ConvertDCSocket(int iSocket)
{
	X86Regs r;

	r.bx = iSocket;
	r.ax = CONVERT_DC_SOCKET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iSocket
Socket handle for the connection.

	Description
Closes the STREAM (TCP) connection (sends a FIN). After EofSocket() has been called, no WriteSocket() calls may be made. The socket, however, remains open for reading until the peer closes the connection.

Returns
On success, 0.
On error, -1 with SocketsErrNo containing the error code.
*/

int EofSocket(int iSocket)
{
	X86Regs r;

	r.bx = iSocket;
	r.ax = EOF_SOCKET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iSocket
Socket handle for the connection.

Description
Flushes any output data still queued for a TCP connection.

Returns
On success, 0
On error, -1 with SocketsErrNo containing the error code.
*/

int FlushSocket(int iSocket)
{
	X86Regs r;

	r.bx = iSocket;
	r.ax = FLUSH_SOCKET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iSocket
Socket handle for the connection.

	Description
Closes connection and release all resources. On a STREAM (TCP) connection, this function should only be called once the connection has been closed from both sides otherwise a reset (ungraceful close) can result.
Returns
On success, socket handle.
On error, -1 with SocketsErrNo containing the error code.
*/

int ReleaseSocket(int iSocket)
{
	X86Regs r;

	r.bx = iSocket;
	r.ax = RELEASE_SOCKET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iSocket
Socket handle for the connection.

	Description
Closes all connections and releases all resources associated with Dos Compatible sockets.

Returns
On success, 0.
On error, -1 with SocketsErrNo containing the error code.
*/
int ReleaseDCSockets(void)
{
	X86Regs r;

	r.ax = RELEASE_DC_SOCKETS << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iSocket
Socket handle for the connection.

	Description
Aborts the network connection and releases all resources.  This function causes an unpredictable close (reset) on a STREAM connection.

Returns
On success, returns socket handle.
On error, -1 with SocketsErrNo containing the error code.
*/
int AbortSocket(int iSocket)
{
	X86Regs r;

	r.ax = ABORT_SOCKET << 8;
	r.bx = iSocket;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
	None

	Description
Aborts all Dos Compatible socket connections.

Returns
On success, 0.
On error, -1 with SocketsErrNo containing the error code.

*/
int AbortDCSockets(void)
{
	X86Regs r;

	r.ax = ABORT_DC_SOCKETS << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
	None

	Description
Shuts down the network and unloads the Sockets TCP/IP kernel.

Returns
On success, 0.
On error, -1 with SocketsErrNo containing the error code.
*/
int ShutDownNet(void)
{
	X86Regs r;

	r.ax = SHUT_DOWN_NET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
	iSocket, socket number to check

	Description
	Gets the local IP address of a connection.  In the case of
	a single interface host, this is the IP address of the host.
	In the case of more than one interface, the IP address of
	the interface being used to route the traffic for the
	specific connection is given.

	Returns
	On success, IP address.
	On error, zero with iNetErrNo containg the error code.
*/

DWORD GetAddress(int iSocket)
{
	DWORD dwAddress;
	X86Regs r;

	r.bx = iSocket;
	r.ax = GET_ADDRESS << 8;

	if(-1 == CallSocketsDosApi(&r))
		return 0L;

	dwAddress = (((DWORD) r.dx << 16L) + r.ax);

	return dwAddress;
}


/*
	Parameters
	None

	Description
Gets peer address information on a connected socket.

Returns
On success, 0 and NET_ADDR structure filled in.
On error, -1 with iNetErrNo containing the error
*/
int GetPeerAddress(int iSocket, NET_ADDR *psAddr)
{
	X86Regs r;

	r.ds = REALMODE_SEGMENT(psAddr);
	r.dx = REALMODE_OFFSET(psAddr);

	r.bx = iSocket;
	r.ax = GET_PEER_ADDRESS << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iReserved
Reserved value, set to zero.
bCode
Code specifying kernel info to retrieve:
K_INF_HOST_TABLE	Gets name of file containing host table.
K_INF_DNS_SERVERS	Gets IP addresses of DNS Servers.
K_INF_TCP_CONS	Gets number of Sockets (DC + normal).
K_INF_BCAST_ADDR	Gets broadcast IP address.
K_INF_IP_ADDR	Gets IP address of first interface.
K_INF_SUBNET_MASK	Gets netmask of first interface.
pData
Pointer to data area to receive kernel information.
puSize
Pointer to WORD containing length of data area.

	Description
Gets specified information from the kernel.

Returns
On success returns 0 with data area and size word filled in.
On error, -1 with SocketsErrNo containing the error on failure.
*/
int GetKernelInformation(int iReserved, BYTE bCode, BYTE bDevID,
						 void *pData, WORD *pwSize)
{
        X86Regs r;


	/*
		Pointers within this module may or may not be far
		depending on the memory model used by the compiler.
		The following code will ensure that the far pointer
		passed into Sockets is entirely zero if the offset
		of the pointer is zero.
	*/
	r.si = REALMODE_OFFSET(pData);
	if(pData)
		r.ds = REALMODE_SEGMENT(pData);
	else
		r.ds = 0;

	r.di = REALMODE_OFFSET(pwSize);
	if(pwSize)
		r.es = REALMODE_SEGMENT(pwSize);
	else
		r.es = 0;

	r.bx = iReserved;
	r.dx = ((WORD)bCode << 8) + bDevID;
	r.ax = GET_KERNEL_INFO << 8;

	return CallSocketsDosApi(&r);
}

/*
	Parameters
dwHost
IP address of host to ping.
iLength
Number of data bytes in ping request.

	Description
Sends an ICMP ping (echo request) and waits until a response is received or for six seconds if no response is received.  ICMPPing() is always a blocking function.

Returns
On success, 0.
On error, -1 with SocketsErrNo containing the error code.
*/
int ICMPPing(DWORD dwHost, int iLength)
{
	X86Regs r;

	r.bx = (WORD) (dwHost & 0xFFFFL);
	r.dx = (WORD) (dwHost >> 16);

	r.cx = iLength;
	r.ax = ICMP_PING << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
psKc
Pointer to KERNEL_CONFIG structure.
bKMaxTcp	Number of TCP sockets allowed.
BKMaxUdp	Number of UDP sockets allowed.
bKMaxIp	Number of IP sockets allowed (0).
bKMaxRaw	Number of RAW_NET sockets allowed (0).
bKActTcp	Number of TCP sockets in use.
bKActUdp	Number of UDP sockets in use.
bKActIp	Number of IP sockets in use (0).
bKActRaw	Number of RAW_NET sockets in use (0).
wKActDCS	Number of active Dos Compatible Sockets.
wKActSoc	Number of active normal Sockets.
bKMaxLnh	Maximum header on an attached network.
bKMaxLnt	Maximum trailer on an attached network.
bKLBUF_SIZE	Size of a large packet buffer.
bKNnet	Number of network interfaces attached.
dwKCticks	Milliseconds since kernel started.
dwKBroadcast	IP broadcast address in use.

	Description
Gets the kernel configuration.

Returns
On success, 0 with KERNEL_CONFIG structure filled in.
On error,  -1 with SocketsErrNo containing the error code.
*/
int GetKernelConfig(KERNEL_CONFIG *psKC)
{
	X86Regs r;

	r.ds = REALMODE_SEGMENT(psKC);
	r.si = REALMODE_OFFSET(psKC);

	r.ax = GET_KERNEL_CONFIG << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iSocket
Socket handle for the connection.
psNI
Pointer to NET_INFO structure.  The following members of NET_INFO are obtained:
DwIPAddress
dwIPSubnet
iUp
iLanLen
pLanAddr

	Description
Gets information about the network.

Returns
On success, 0 with NET_INFO structure filled .
On erro, -1 with SocketsErrNo containing the error code.
*/
int GetNetInfo(int iSocket, NET_INFO *psNI)
{
	X86Regs r;

	r.ds = REALMODE_SEGMENT(psNI);
	r.si = REALMODE_OFFSET(psNI);

	r.bx = iSocket;
	r.ax = GET_NET_INFO << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iSocket
Socket handle for the connection.
iType
Type of connection: STREAM or DATAGRAM.
psAddr
Pointer to NET_ADDR structure.

	Description
Makes a network connection.  If iSocket is specified as -1, a Dos Compatible socket is assigned.  In this case only, DOS is called to open a file handle.  If iSocket specifies a non-blocking socket or iType specifies a DATAGRAM connection, this call returns immediately.  In the case of a STREAM connection, the connection may not yet be established.  ReadSocket() can be used to test for connection establishment.  As long as ReadSocket() returns an ERR_NOT_ESTAB code, the connection is not established.  A good return or an error return with ERR_WOULD_BLOCK indicates an establishedconnection.  A more complex method uses SetAsyncNotify() with NET_AS_OPEN to test for connection establishment.  NET_AS_ERROR should also be set to be notified of a failed open attempt.

Returns
On success, socket.
On error, -1 with SocketsErrNo containing the error code.
*/
int ConnectSocket(int iSocket, int iType, NET_ADDR *psAddr)
{
	X86Regs r;

	psAddr->bProtocol = 6;

	r.ds = REALMODE_SEGMENT(psAddr);
	r.si = REALMODE_OFFSET(psAddr);

	r.bx = iSocket;
	r.dx = iType;
	r.ax = CONNECT_SOCKET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iSocket
Socket handle for the connection.
iType
Type of connection:  STREAM or DATAGRAM.
psAddr
Pointer to NET_ADDR structure.

	Description
Listens for a network connection. If iSocket is specified as -1, a Dos Compatible socket is assigned. In this case only, DOS is called to open a file handle.  If iSocket specifies a non-blocking socket or iType specifies a DATAGRAM connection, this call returns immediately.  In the case of a STREAM connection, the connection may not be established yet.  ReadSocket() can be used to test for connection establishment.
As long as ReadSocket() returns an ERR_NOT_ESTAB code, the connection is not established.  A good return or an error return with ERR_WOULD_BLOCK indicates connection establishment.  A more complex method is to use SetAsyncNotify() with NET_AS_OPEN to test for connection establishment.  NET_AS_ERROR should also be set to be notified of a failed open attempt.

Returns
On success, returns socket handle on success.
On error, -1 on failure with SocketsErrNo containing the error code.
*/
int ListenSocket(int iSocket, int iType, NET_ADDR *psAddr)
{
	X86Regs r;

	psAddr->bProtocol = 6;

	r.ds = REALMODE_SEGMENT(psAddr);
	r.si = REALMODE_OFFSET(psAddr);

	r.bx = iSocket;
	r.dx = iType;
	r.ax = LISTEN_SOCKET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iMaxid
Number of sockets to test.
plIflags
Pointer to input flags indicating receive data availability.
plOflags
Pointer to output flags indicating readiness to write.

	Description
Tests all Dos Compatible sockets for data availability and readiness to write.  A 32-bit DWORD representing 32 DC sockets is filled in for each socket with receive data, and another 32-bit DWORD for DC sockets ready for writing.  The least-significant bit represents the socket with value 0 and the most-significant bit represents the socket with value 31.  Bits representing unused sockets are left unchanged.

Returns
On success, returns 0 with *plIflags and *plOflags filled in with current status.
On error, -1 on failure with SocketsErrNo containing the error code.
*/
int SelectSocket(int iMaxID, long *plIflags, long *plOflags)
{
	X86Regs r;

	r.ds = REALMODE_SEGMENT(plIflags);
	r.dx = REALMODE_OFFSET(plIflags);
	r.es = REALMODE_SEGMENT(plOflags);
	r.di = REALMODE_OFFSET(plOflags);

	r.bx = iMaxID;
	r.ax = SELECT_SOCKET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iSocket
Socket handle for the connection.
pcBuf
Pointer to buffer to receive data.
wLen
Length of buffer, ie maximum number of bytes to read.
psFrom
Pointer to NET_ADDR structure to receive address information about local and remote ports and remote IP address.
wFlags
Flags governing operation. Any combination of:
NET_FLG_PEEK	Don't dequeue data.
NET_FLG_NON_BLOCKING	Don't block.

	Description
Reads from the network using a socket.  Returns as soon as any non-zero amount of data is available, regardless of the blocking state.  If the operation is non-blocking, either by having used SetSocketOption() with the NET_OPT_NON_BLOCKING option or specifying wFlags with NET_FLG_NON_BLOCKING, ReadSocket() returns immediately with the count of available data or an error of ERR_WOULD_BLOCK.
In the case of a STREAM (TCP) socket, record boundaries do not exist and any amount of data can be read at any time regardless of the way it was sent by the peer.  No data is truncated or lost even if more data than the buffer size is available.  What is not returned on one call, is returned on subsequent calls.  If a NULL buffer is specified, the number of bytes on the receive queue is returned.
In the case of a DATAGRAM (UDP) socket, the entire datagram is returned in one call, unless the buffer is too small in which case the data is truncated, thereby preserving record boundaries.  Truncated data is lost. If data is available and both the NET_FLG_PEEK and NET_FLG_NON_BLOCKING flags are specified, the number of datagrams on the receive queue is returned.  If data is available and NET_FLG_PEEK is set and a NULL buffer is specified, the number of bytes in the next datagram is returned.

Returns
On success, returns number of bytes read.
On error, -1 on failure with SocketsErrNo containing the error code. A return code of 0 indicates that the peer has closed the connection.
*/
int ReadSocket(int iSocket, char *pcBuf, WORD wLen,
			   NET_ADDR *psFrom, WORD wFlags)
{
	X86Regs r;

	/*
		Pointers within this module may or may not be far
		depending on the memory model used by the compiler.
		The following code will ensure that the far pointer
		passed into Sockets is entirely zero if the offset
		of the pointer is zero.
	*/
	r.di = REALMODE_OFFSET(psFrom);
	if(psFrom)
		r.es = REALMODE_SEGMENT(psFrom);
	else
		r.es = 0;

	r.si = REALMODE_OFFSET(pcBuf);
	if(pcBuf)
		r.ds = REALMODE_SEGMENT(pcBuf);
	else
		r.ds = 0;

	r.dx = wFlags;
	r.cx = wLen;
	r.bx = iSocket;
	r.ax = READ_SOCKET << 8;

	if (-1 == CallSocketsDosApi(&r))
		return -1;
	else
		return r.cx;
}

/*
	Parameters
iSocket
Socket socket for the connection.
pcBuf
Pointer to buffer to receive data.
wLen
Length of buffer, ie maximum number of bytes to read.
psFrom
Pointer to NET_ADDR structure to receive address information about local and remote ports and remote IP address.
wFlags
Flags governing operation. Any combination of:
NET_FLG_PEEK	Don't dequeue data.
NET_FLG_NON_BLOCKING	Don't block.

	Description
Reads from the network using a socket and is only intended to be used on DATAGRAM sockets. All datagrams from the IP address and port matching the values in the NET_ADDR structure are returned while others are discarded. A zero value for dwRemoteHost is used as a wildcard to receive from any host and a zero value for wRemotePort is used as a wildcard to receive from any port. The local port, wLocalPort , can not be specified as zero.
In other respects ReadFromSocket() behaves the same as ReadSocket().

Returns
On success, returns number of bytes read.
On error, -1 with SocketsErrNo containing the error code. A return code of 0 indicates that the peer has closed the connection. Note the following anomaly:
If blocking is disabled, a failure with an error code of ErrWouldBlock is completely normal and only means that no data is currently available.
*/
int ReadFromSocket(int iSocket, char *pcBuf, WORD wLen,
				   NET_ADDR *psFrom, WORD wFlags)
{
	X86Regs r;

	/*
		Pointers within this module may or may not be far
		depending on the memory model used by the compiler.
		The following code will ensure that the far pointer
		passed into Sockets is entirely zero if the offset
		of the pointer is zero.
	*/
	r.di = REALMODE_OFFSET(psFrom);
	if(psFrom)
		r.es = REALMODE_SEGMENT(psFrom);
	else
		r.es = 0;

	r.si = REALMODE_OFFSET(pcBuf);
	if(pcBuf)
		r.ds = REALMODE_SEGMENT(pcBuf);
	else
		r.ds = 0;

	r.dx = wFlags;
	r.cx = wLen;
	r.bx = iSocket;
	r.ax = READ_FROM_SOCKET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iSocket
Socket handle for the connection.
pcBuf
Pointer to buffer to containing send data.
wLen
Length of buffer, ie number of bytes to write.
wFlags
Flags governing operation; can be any combination of:
NET_FLG_OOB	Send out of band data (TCP only).
NET_FLG_PUSH	Disregard Nagle heuristic (TCP only).
NET_FLG_NON_BLOCKING	Don't block.
NET_FLG_BROADCAST	Broadcast data (UDP only).
NET_FLG_MC_NOECHO	Suppress the local echo of a multicast datagram.

	Description
Writes to the network using a socket.

Returns
On success, returns number of bytes written.
On error, -1 on failure with SocketsErrNo containing the error code.
The number of bytes actually written on a non-blocking write, can be less than wLen.  In such a case, the writing of the unwritten bytes must be retried, preferably after some delay.
*/
int WriteSocket(int iSocket, char *pcBuf, WORD wLen, WORD wFlags)
{
	X86Regs r;

	r.ds = REALMODE_SEGMENT(pcBuf);
	r.si = REALMODE_OFFSET(pcBuf);

	r.bx = iSocket;
	r.cx = wLen;
	r.dx = wFlags;
	r.ax = WRITE_SOCKET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iSocket
Socket handle for the connection.
pcBuf
Pointer to buffer to containing send data.
wLen
Length of buffer, ie number of bytes to write.
psTo
Pointer to NET_ADDR structure containing local port to write from and remote port and IP address to write to.
wFlags
Flags governing operation. Any combination of:
NET_FLG_NON_BLOCKING	Don't block.
NET_FLG_BROADCAST	Broadcast data (UDP only).

	Description
Writes to the network using a network address (UDP only).

Returns
On success, returns number of bytes written.
On error, -1 with SocketsErrNo containing the error code.
*/
int WriteToSocket(int iSocket, char *pcBuf, WORD wLen,
				  NET_ADDR *psTo, WORD wFlags)
{
	X86Regs r;

	r.ds = REALMODE_SEGMENT(pcBuf);
	r.si = REALMODE_OFFSET(pcBuf);
	r.es = REALMODE_SEGMENT(psTo);
	r.di = REALMODE_OFFSET(psTo);

	r.bx = iSocket;
	r.cx = wLen;
	r.dx = wFlags;
	r.ax = WRITE_TO_SOCKET << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
iSocket
Socket handle for the connection.
dwTime
Timer delay in milliseconds.
lpHandler
Far address of alarm callback.  See the description of  SetAsyncNotification() for the format of the callback function.
dwHint
Argument to be passed to callback function.

	Description
The SetAlarm() function sets an alarm timer.

Returns
On success, returns socket handle.
On error, -1 with SocketsErrNo containing the error code.
*/
int SetAlarm(int iSocket, DWORD dwTime,
			 int (D_FAR *lpHandler)(), DWORD dwHint)
{
	X86Regs r;

	r.ds = REALMODE_SEGMENT(lpHandler);
	r.si = REALMODE_OFFSET(lpHandler);

	r.cx = (WORD) (dwTime >> 16);
	r.dx = (WORD) (dwTime & 0xFFFFL);

	r.es = (WORD) (dwHint >> 16);
	r.di = (WORD) (dwHint & 0xFFFFL);

	r.bx = iSocket;
	r.ax = SET_ALARM << 8;

	return CallSocketsDosApi(&r);
}

/*
	Parameters
iSocket
Socket handle for the connection.
iEvent
Event which is being set:
NET_AS_OPEN	Connection has opened.
NET_AS_RCV	Data has been received.
NET_AS_XMT	Ready to transmit.
NET_AS_FCLOSE	Peer has closed connection.
NET_AS_CLOSE	Connection has been closed.
NET_AS_ERROR	Connection has been reset.
lpHandler
Far address of callback function.
dwHint
Argument to be passed to callback function
The handler is not compatible with C calling conventions but is called by a far call with the following parameters:
BX = Socket handle.
CX = Event.
ES:DI = dwHint argument passed to SetAsyncNotification() or SetAlarm().
DS:DX = SI:DX = variable argument depending on event:
NET_AS_OPEN
NET_AS_CLOSE	Pointer to NET_ADDR address structure.
NET_AS_FCLOSE
NET_AS_RCV
NET_AS_ALARM	Zero.
NET_AS_XMT	Byte count which can be sent without blocking.
NET_AS_ERROR	Error code -ERR_TERMINATING, ERR_TIME_OUT or ERR_RESET.

	Description
Sets an asynchronous notification (callback) for a specific event.
Other CAPI functions may be called in the callback, with the exception of ResolveName() which may call DOS.  The callback is not compatible with C argument-passing conventions and some care must be taken.  Some CPU register manipulation is required.  This can be done by referencing CPU registers, such as _BX, or by means of assembler instructions.
In the callback, the stack is supplied by Sockets and may be quite small depending on the /s= command line option when loading Sockets.  The stack segment is obviously not equal to the data segment, which can cause problems when the Tiny, Small or Medium memory model is used.  The simplest way to overcome the problem is to use the Compact, Large or Huge memory model.  Other options - use the DS != SS compiler option or do a stack switch to a data segment stack .
If the callback is written in C or C++, the _loadds modifier can be used to set the data segment to that of the module, which destroys the DS used for the variable argument.  (This is why DS == SI on entry for Sockets version 1.04 and later.)  An alternate method is to use the argument passed to SetAsyncNotification() in ES:DI as a pointer to a structure that is accessible from both the main code and the callback.  If DS is not set to the data segment of the module, then the functions in CAPI.C do not work: Don't use them in the callback.
The callback will probably be performed at interrupt time with no guarantee of reentry to DOS.  Do not use any function, such as putchar() or printf(), in the callback which may cause DOS to be called.
It is good programming practice to do as little as possible in the callback.  The setting of event flags which trigger an operation at a more stable time is recommended.
Callback functions do not nest.  The callback function is not called while a callback is still in progress, even if other CAPI functions are called.
To alleviate the problems in items 2, 3 and 4 above, a handler is provided in CAPI.C which uses the dwHint parameter to pass the address of a C-compatible handler, with a stack which is also C-compatible.  This handler is named AsyncNotificationHandler.  A user handler named MyHandler below, is called in the normal way with a stack of 500 bytes long.  The stack size value can be set by changing the HANDLER_STACK_SIZE constant in CAPI.C.

Returns
On success, returns pointer to the previous callback handler.
On error, -1 with SocketsErrNo containing the error code.
*/
int D_FAR *SetAsyncNotification(int iSocket, int iEvent,
								int (D_FAR *lpHandler)(), DWORD dwHint)
{
	X86Regs r;

	r.ds = REALMODE_SEGMENT(lpHandler);
	r.dx = REALMODE_OFFSET(lpHandler);

	r.es = (WORD) (dwHint >> 16);
	r.di = (WORD) (dwHint & 0xFFFFL);

	r.bx = iSocket;
	r.cx = iEvent;
	r.ax = SET_ASYNC_NOTIFICATION << 8;

	if(-1 == CallSocketsDosApi(&r))
	{
		return (int D_FAR *) MAKE_REAL_POINTER(-1, -1);
	}

	return (int D_FAR *) MAKE_REAL_POINTER(r.ds, r.dx);
}

/*
	Parameters
pszName
Pointer to string containing symbolic name.
pcCname
Pointer to buffer to receive canonical name.
ICnameLen
Length of buffer pointed to by pcName.

	Description
Resolves IP address from symbolic name.

Returns
On success, returns IP address.
On failure, 0 with SocketsErrNo containing the error code.
*/
DWORD ResolveName(char *pszName, char *pszCName, int iCNameLen)
{
	X86Regs r;

	/*
		Pointers within this module may or may not be far
		depending on the memory model used by the compiler.
		The following code will ensure that the far pointer
		passed into Sockets is entirely zero if the offset
		of the pointer is zero.
	*/
	r.dx = REALMODE_OFFSET(pszName);
	if(pszName)
		r.ds = REALMODE_SEGMENT(pszName);
	else
		r.ds = 0;

	r.di = REALMODE_OFFSET(pszCName);
	if(pszCName)
		r.es = REALMODE_SEGMENT(pszCName);
	else
		r.es = 0;

	r.cx = iCNameLen;
	r.ax = RESOLVE_NAME << 8;

	if(-1 == CallSocketsDosApi(&r))
		return 0L;
	else
		return (((DWORD)r.dx << 16L) + r.ax);
}

/*
	Parameters
pszName
Pointer to string containing dotted decimal address.

	Description
Gets IP address from dotted decimal address.

Returns
On success, returns IP address.
On error, 0 with SocketsErrNo containing the error code.
*/
DWORD ParseAddress(char *pszName)
{
	X86Regs r;

	r.ds = REALMODE_SEGMENT(pszName);
	r.dx = REALMODE_OFFSET(pszName);

	r.ax = PARSE_ADDRESS << 8;

	if(-1 == CallSocketsDosApi(&r))
		return 0L;
	else
		return (((DWORD) r.dx << 16L) + r.ax);
}

/*
	Parameters
iSocket
Socket handle for the connection.
iLevel
Level of option. This value is ignored.
iOption
Option to set.
NET_OPT_NON_BLOCKING	Set blocking off if dwOptionValue is non-zero.
NET_OPT_TIMEOUT	Set the timeout to dwOptionValue milliseconds. Turn off timeout if dwOptionValue is zero.
NET_OPT_WAIT_FLUSH	Wait for flush if dwOptionValue is non-zero.
dwOptionValue
Option value.
iLen
Length of dwOptionValue, 4 in all cases.

	Description
Sets an option on the socket.

Returns
On success, returns global socket.
On error, -1 with SocketsErrNo containing the error code.
*/
int SetSocketOption(int iSocket, int iLevel, int iOption,
					DWORD dwOptionValue, int iLen)
{
	X86Regs r;

	r.ds = (WORD) (dwOptionValue >> 16);
	r.dx = (WORD) (dwOptionValue & 0xFFFFL);

	r.bx = iSocket;
	r.cx = iLen;
	r.si = iLevel;
	r.di = iOption;
	r.ax = SET_OPTION << 8;

	return CallSocketsDosApi(&r);
}

/*
	Parameters
dwGroupAddress
The group address on which to receive multicast datagrams.
dwIPAddress
The IP adress for the interface to use.  The first interface to be specified in SOCKET.CFG is the default interface in the case where dwIPAddress == 0.

	Description
Causes Sockets to join a multicast group.

Returns
On success, 0.
On error, any other integer value contains the error code.
*/
int JoinGroup(DWORD dwGroupAddress, DWORD dwInterfaceAddress)
{
	X86Regs r;
	GROUP_ADDR sGroupAd;
	GROUP_ADDR *psGroupAd = &sGroupAd ;

	sGroupAd.dwGroupAddr = dwGroupAddress;
	sGroupAd.dwIFAddr = dwInterfaceAddress;

	r.ds = REALMODE_SEGMENT(psGroupAd);
	r.si = REALMODE_OFFSET(psGroupAd);

	r.ax = JOIN_GROUP << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
dwGroupAddress
The group address on which multicast datagrams are being received.
dwIPAddress
The IP adress for the interface being used.  The first interface to be specified in SOCKET.CFG is the default interface in the case where dwIPAddress == 0.

	Description
The LeaveGroup() function causes Sockets to leave a multicast group.

Returns
On success, returns 0. 
On error, any other integer value contains the error code.
*/
int LeaveGroup(DWORD dwGroupAddress, DWORD dwInterfaceAddress)
{
	X86Regs r;
	GROUP_ADDR sGroupAd;
	GROUP_ADDR *psGroupAd = &sGroupAd;

	sGroupAd.dwGroupAddr = dwGroupAddress;
	sGroupAd.dwIFAddr = dwInterfaceAddress;

	r.ds = REALMODE_SEGMENT(psGroupAd);
	r.si = REALMODE_OFFSET(psGroupAd);

	r.ax = LEAVE_GROUP << 8;
	return CallSocketsDosApi(&r);
}

/*
	Parameters
	pszName
		Pointer to interface name.
	WFunction
		Function to perform:
			IOCTL_CONNECT	Start dial operation
			IOCTL_DISCONNECT	Disconnect modem
			IOCTL_ENABLEPORT	Enable communications port
			IOCTL_DISABLEPORT	Disable communications port
			IOCTL_ENABLEDOD	Enable dial-on-demand
			IOCTL_DISABLEDOD	Disable dial-on-demand
			IOCTL_GETSTATUS	Get modem/connection status

	Description
	The IfaceIOCTL function controls asynchronous interfaces.
	
	Returns
	On success, returns >=0.
	On error, returns -1.
	IOCTL_GETSTATUS returns the following bits:
		ST_DTR	0x01	Data Terminal Ready
		ST_RTS	0x02	Request To Send
		ST_CTS	0x10	Clear To Send
		ST_DSR	0x20	Data Set Ready
		ST_RI	0x40	Ring Indicator
		ST_DCD	0x80	Data Carrier Detect
		ST_CONNECTED	0x100	Modem is connected
		ST_MODEMSTATE	0xe00	Modem state mask
		STM_NONE	0x000	No modem on port
		STM_IDLE	0x200	Modem is idle
		STM_INITIALIZING	0x400	Modem is initializing
		STM_DIALING	0x600	Modem is dialing
		STM_CONNECTING	0x800	Modem is connecting
		STM_ANSWERING	0xa00	Modem is answering
*/
int IfaceIOCTL(char *pszName, WORD wFunction)
{
	X86Regs r;

	r.ds = REALMODE_SEGMENT(pszName);
	r.si = REALMODE_OFFSET(pszName);

	r.ax = (IFACE_IOCTL << 8) + wFunction & 0x00FF;

	return CallSocketsDosApi(&r);
}







